% epsilon_estimate_bayes.m
% 
% Estimate eps (elasticity of substitution) using limited information 
%       Bayes estimates
% 
% "The Past and Future of U.S. Structural Change" 
% Andrew Foerster, Andreas Hornstein, Pierre-Daniel Sarte, Mark Watson
% September 2025
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

% -- Clear Workspace -- %
restoredefaultpath;
clear;
clc;
close all;

% -- File Directories -- % 
datadir = 'Data\DataReplication\DataFinal\'; 
outdir  = 'Tables\';
figdir  = 'Figures\';
matdir  = 'Output\';

% -- Add paths -- %
addpath('Functions\');
addpath('Utilities\')
addpath('Data\')

% Set random number seed
rng(6122137);   % For replication

% -- Load Data and Set up -- %
load_data;
period_lower = 20;       % Shortest Period 
Share_mat = sV_mat;
T = size(Share_mat,1);
% Compute q
q = floor(2*T/period_lower);

% Log Prices
py = y ./ yq;
log_py = log(py);
d_p = dif(log_py,1);

% Log Shares of Investment
log_omega = log(omega);
% Log Shares of Materials
log_phi = log(phi);

% Durables and IPP in Sector 
P_Dur = py(1:T,2);
P_IPP = py(1:T,3);
S_X_Dur = squeeze(omega(2,:,:))';
S_X_IPP = squeeze(omega(3,:,:))';
S_M_Dur = squeeze(phi(2,:,:))';
S_M_IPP = squeeze(phi(3,:,:))';
X_Dur_IPP = (S_X_Dur./repmat(P_Dur,1,5))./(S_X_IPP./repmat(P_IPP,1,5)); % Relative quantities of dur/ipp in investment
M_Dur_IPP = (S_M_Dur./repmat(P_Dur,1,5))./(S_M_IPP./repmat(P_IPP,1,5)); % Relative quantities of dur/ipp in materials   
P_Dur_IPP = P_Dur./P_IPP; % Relative prices of dur/ipp

% Compute logs 
x_Dur_IPP = log(X_Dur_IPP);
m_Dur_IPP = log(M_Dur_IPP);
p_Dur_IPP = log(P_Dur_IPP);
% Compute first differences
d_x_Dur_IPP = dif(x_Dur_IPP,1);
d_m_Dur_IPP = dif(m_Dur_IPP,1);
d_p_Dur_IPP = dif(p_Dur_IPP,1);
d_a_Dur_IPP = d_tfp(:,2)-d_tfp(:,3); % Relative TFP growth rates

% Drop first observation, which is missing
d_x_Dur_IPP = d_x_Dur_IPP(2:end,:);
d_m_Dur_IPP = d_m_Dur_IPP(2:end,:);
d_p_Dur_IPP = d_p_Dur_IPP(2:end,:);
d_a_Dur_IPP = d_a_Dur_IPP(2:end,:);
calvec = calvec(2:end);
calds = calds(2:end,:);
T = length(calvec);

% Get Cosine transforms
[cx,~,~,~] = cos_transform(d_x_Dur_IPP,q);
[cm,~,~,~] = cos_transform(d_m_Dur_IPP,q);
[cp,~,~,~] = cos_transform(d_p_Dur_IPP,q);
[ca,~,~,~] = cos_transform(d_a_Dur_IPP,q);

% Priors
%  eps
mu_eps_prior = 1.0;
var_eps_prior = (2.0)^2;

%  pia
mu_pia_prior = 0;
var_pia_prior = (50)^2;  % Diffuse
% Covariance matrix
nu_prior = 0.01;
V_prior = eye(2);

% Gibbs Sampling reps% Number of draws
n_burnin = 10000;        % Discarded Draws
n_draws_save = 2500;     % Number of Draws to Save
k_draws = 1000;          % Save results every k_draws

eps_draws = NaN(n_draws_save,5);
% Loop over sectors
for i = 1:5
    disp(['Sector: ' num2str(i)])
    tic;
    % Data
    qrat = cm(:,i); % Relative quantities for Materials
    prat = cp;
    arat = ca;
    % Intialize parameters
    eps = 1.0;
    pia = -0.5;
    sigma = [1 0;0 1];
    % Indices for saving draws
    itmp = 0;
    tic;
    kk_draw  = 0;
    jj_draw = 0;
    for idraw = 1:n_burnin+n_draws_save*k_draws
        % Draw Sigma
        sigma = draw_sigma(qrat,prat,arat,eps,pia,nu_prior,V_prior);
        % Draw eps value
        eps = draw_eps(qrat,prat,arat,pia,sigma,mu_eps_prior,var_eps_prior);
        % Draw pia value
        pia = draw_pia(qrat,prat,arat,eps,sigma,mu_pia_prior,var_pia_prior);
        
        % Save Draws;
        if idraw > n_burnin
            kk_draw = kk_draw+1;
            if kk_draw == k_draws
                jj_draw = jj_draw+1;
                kk_draw = 0;
                eps_draws(jj_draw,i) = eps;
            end
        end      
    end
    toc;
end
eps_draws_m = eps_draws;

eps_draws = NaN(n_draws_save,5);
% Loop over sectors
for i = 1:5
    disp(['Sector: ' num2str(i)])
    tic;
    % Data
    qrat = cx(:,i); % Relative quantities for Materials
    prat = cp;
    arat = ca;
    % Intialize parameters
    eps = 1.0;
    pia = -0.5;
    sigma = [1 0;0 1];
    % Indices for saving draws
    itmp = 0;
    tic;
    kk_draw  = 0;
    jj_draw = 0;
    for idraw = 1:n_burnin+n_draws_save*k_draws
        % Draw Sigma
        sigma = draw_sigma(qrat,prat,arat,eps,pia,nu_prior,V_prior);
        % Draw eps value
        eps = draw_eps(qrat,prat,arat,pia,sigma,mu_eps_prior,var_eps_prior);
        % Draw pia value
        pia = draw_pia(qrat,prat,arat,eps,sigma,mu_pia_prior,var_pia_prior);
        
        % Save Draws;
        if idraw > n_burnin
            kk_draw = kk_draw+1;
            if kk_draw == k_draws
                jj_draw = jj_draw+1;
                kk_draw = 0;
                eps_draws(jj_draw,i) = eps;
            end
        end      
    end
    toc;
end
eps_draws_x = eps_draws;

% Save Results
save([matdir 'eps_draws_' num2str(q) '.mat'],'eps_draws_x','eps_draws_m');

% Functions 
function sigma = draw_sigma(q,p,a,eps,pia,nu_prior,V_prior)
    e_q = -q - p*eps;
    e_p = p - a*pia;
    sigma = draw_sigma_iw([e_q e_p],nu_prior,V_prior);
end

function eps = draw_eps(q,p,a,pia,sigma,mu_eps_prior,var_eps_prior)
    e_p = p - a*pia;
    b = sigma(1,2)/sigma(2,2);
    var_e = sigma(1,1)-b*sigma(1,2);
    ytilde = -q - e_p*b;
    [mu_post,var_post] = lin_model(ytilde,p,mu_eps_prior,var_eps_prior,var_e);
    i_pos = 0;
    while i_pos == 0
        eps = mu_post + sqrt(var_post)*randn(1);
        i_pos = eps > 0;
    end
end

function pia = draw_pia(q,p,a,eps,sigma,mu_pia_prior,var_pia_prior)
    e_q = -q - p*eps;
    b = sigma(1,2)/sigma(1,1);
    var_e = sigma(2,2)-b*sigma(1,2);
    ytilde = p - e_q*b;
    [mu_post,var_post] = lin_model(ytilde,a,mu_pia_prior,var_pia_prior,var_e);
    pia = mu_post + sqrt(var_post)*randn(1);
end

function [mu_b_post,var_b_post] = lin_model(y,x,mu_b_prior,var_b_prior,var_eps)
    bhat = x\y;
    v_bhat = var_eps/(x'*x);
    k = var_b_prior/(v_bhat + var_b_prior);
    mu_b_post = k*bhat + (1-k)*mu_b_prior;
    var_b_post = var_b_prior - k*var_b_prior;
end


function [sigma_draw] = draw_sigma_iw(y,nu_prior,s2_prior)
    % Draw Sigma using inverse Wishart prior
    %
    T = size(y,1);
    %n = size(y,2);
    SSR_mat = y'*y;
    SSR_prior = nu_prior*s2_prior;
    SSR = SSR_mat + SSR_prior;
    nu = T + nu_prior;
    sigma_draw = iwishrnd(SSR,nu);
end
